<?php

#
# ss_nut_ups_status.php
# version 0.6
# November 11, 2010
#
# Copyright (C) 2007-2010, Eric A. Hall
# http://www.eric-a-hall.com/
#
# This software is licensed under the same terms as Cacti itself
#

#
# load the Cacti configuration settings if they aren't already present
#
if (isset($config) == FALSE) {

	if (file_exists(dirname(__FILE__) . "/../include/config.php")) {
		include_once(dirname(__FILE__) . "/../include/config.php");
	}

	if (file_exists(dirname(__FILE__) . "/../include/global.php")) {
		include_once(dirname(__FILE__) . "/../include/global.php");
	}

	if (isset($config) == FALSE) {
		echo ("FATAL: Unable to load Cacti configuration files \n");
		return;
	}
}

#
# call the main function manually if executed outside the Cacti script server
#
if (isset($GLOBALS['called_by_script_server']) == FALSE) {

	array_shift($_SERVER["argv"]);
	print call_user_func_array("ss_nut_ups_status", $_SERVER["argv"]);
}

#
# main function
#
function ss_nut_ups_status($protocol_bundle="",
	$cacti_request="", $data_request="", $data_request_key="") {

	#
	# 1st function argument contains the protocol-specific bundle
	#
	# use '====' matching for strpos in case colon is 1st character
	#
	if ((trim($protocol_bundle) == "") || (strpos($protocol_bundle, ":") === FALSE)) {

		echo ("FATAL: No NUT parameter bundle provided\n");
		ss_nut_ups_status_syntax();
		return;
	}

	$protocol_array = explode(":", $protocol_bundle);

	if (count($protocol_array) < 3) {

		echo ("FATAL: Not enough elements in NUT parameter bundle\n");
		ss_nut_ups_status_syntax();
		return;
	}

	if (count($protocol_array) > 3) {

		echo ("FATAL: Too many elements in NUT parameter bundle\n");
		ss_nut_ups_status_syntax();
		return;
	}

	#
	# 1st bundle element is $nut_hostname
	#
	$nut_hostname = trim($protocol_array[0]);

	if ($nut_hostname == "") {

		echo ("FATAL: Hostname not specified in NUT parameter bundle\n");
		ss_nut_ups_status_syntax();
		return;
	}

	#
	# 2nd bundle element is $nut_port
	#
	$nut_port = trim($protocol_array[1]);

	if ($nut_port == "") {

		#
		# if the value was omitted use the default port number
		#
		$nut_port = 3493;
	}

	if (is_numeric($nut_port) == FALSE) {

		echo ("FATAL: Non-numeric port number \"$nut_port\" specified in NUT parameter bundle\n");
		ss_nut_ups_status_syntax();
		return;
	}

	#
	# 3rd bundle element is $nut_timeout
	#
	$nut_timeout = trim($protocol_array[2]);

	if ($nut_timeout == "") {

		#
		# if the value was omitted use the global default timeout
		#
		$nut_timeout = read_config_option("snmp_timeout");

		#
		# Cacti uses milliseconds but php sockets uses seconds so divide timeout by 1000
		#
		$nut_timeout = ($nut_timeout / 1000);
	}

	if (is_numeric($nut_timeout) == FALSE) {

		echo ("FATAL: Non-numeric NUT timeout \"$nut_timeout\" specified in NUT parameter bundle\n");
		ss_nut_ups_status_syntax();
		return;
	}

	#
	# set minimum value of "1" second
	#
	if ($nut_timeout < 1) {

		$nut_timeout = 1;
	}

	#
	# 2nd function argument is $cacti_request
	#
	$cacti_request = strtolower(trim($cacti_request));

	if ($cacti_request == "") {

		echo ("FATAL: No Cacti request provided\n");
		ss_nut_ups_status_syntax();
		return;
	}

	if (($cacti_request != "index") &&
		($cacti_request != "query") &&
		($cacti_request != "get")) {

		echo ("FATAL: \"$cacti_request\" is not a valid Cacti request\n");
		ss_nut_ups_status_syntax();
		return;
	}

	#
	# remaining function arguments are $data_request and $data_request_key
	#
	if (($cacti_request == "query") || ($cacti_request == "get")) {

		$data_request = strtolower(trim($data_request));

		if ($data_request == "") {

			echo ("FATAL: No data requested for Cacti \"$cacti_request\" request\n");
			ss_nut_ups_status_syntax();
			return;
		}

		if (($data_request != "ups.device") &&
			($data_request != "ups.description") &&
			($data_request != "battery.charge") && 
			($data_request != "input.voltage") &&
			($data_request != "input.frequency") &&
			($data_request != "battery.voltage") &&
			($data_request != "ups.temperature") &&
			($data_request != "ups.load")) {
	
			echo ("FATAL: \"$data_request\" is not a valid data request\n");
			ss_nut_ups_status_syntax();
			return;
		}
	
		#
		# get the index variable
		#
		if ($cacti_request == "get") {

			$data_request_key = strtolower(trim($data_request_key));

			if ($data_request_key == "") {

				echo ("FATAL: No index value provided for \"$data_request\" data request\n");
				ss_nut_ups_status_syntax();
				return;
			}
		}

		#
		# clear out spurious command-line parameters on query requests
		#
		else {
			$data_request_key = "";
		}
	}

	#
	# establish the raw TCP session
	#
	# use '====' matching for session handle to avoid matching zero
	#
	$nut_session = fsockopen($nut_hostname, $nut_port, $errno, $errstr, $nut_timeout);

	if ($nut_session === FALSE) {

		echo ("FATAL: Unable to establish TCP session with port $nut_port on $nut_hostname\n");
		return;
	}

	#
	# set the default socket timeout
	#
	stream_set_timeout($nut_session, $nut_timeout);

	#
	# if they want data for just one ups device, use the input data to seed the array
	#
	if ($cacti_request == "get") {

		#
		# probe for the description associated with the requested device
		#
		fwrite($nut_session, "GET UPSDESC $data_request_key \n");

		#
		# use regex to locate the UPS name and description
		#
		# exit if no match found
		#
		$nut_response = fgets($nut_session, 1024);

		if (preg_match("/^UPSDESC (\S+) \"(.*)\"$/", $nut_response, $scratch) == 0) {

				echo ("FATAL: Unable to locate the requested UPS in NUT output\n");
				return;
		}

		#
		# matches were found so use them for UPS index and description
		#
		$ups_array[0]['index'] = $data_request_key;
		$ups_array[0]['desc'] = trim($scratch[2]);
	}

	#
	# if they want data for all ups devices, use the LIST UPS output to seed the array
	#
	else {
		fwrite($nut_session, "LIST UPS\n");

		#
		# exit if the listing header is not returned first
		#
		if (fgets($nut_session, 1024) != "BEGIN LIST UPS\n") {

			echo "FATAL: Unable to locate UPS devices in NUT output\n";
			return;
		}

		#
		# get the UPS names and descriptions
		#
		# use "===" to avoid false matching eof
		#
		$ups_count = 0;

		while (feof($nut_session) === FALSE) {

			$nut_array[$ups_count] = fgets($nut_session, 1024);

			#
			# look for the end of list
			#
			if ($nut_array[$ups_count] == "END LIST UPS\n") {

				#
				# remove the last entry from the array
				#
				unset ($nut_array[$ups_count]);

				#
				# break out of the loop
				#
				break;
			}

			#
			# use regex to locate the UPS name and description
			#
			# exit if no match found
			#
			if (preg_match("/^UPS (\S+) \"(.*)\"$/", $nut_array[$ups_count], $scratch) == 0) {

				echo ("FATAL: Unable to locate UPS devices in NUT output\n");
				return;
			}

			#
			# matches were found so use them for UPS index and description
			#
			$ups_array[$ups_count]['index'] = trim($scratch[1]);
			$ups_array[$ups_count]['desc'] = trim($scratch[2]);

			#
			# incrmenet the ups counter
			#
			$ups_count++;
		}
	}

	#
	# verify that the ups_array exists and has data
	#
	if ((isset($ups_array) == FALSE) ||
		(count($ups_array) == 0)) {

		echo ("FATAL: No matching UPS devices were returned from NUT\n");
		return;
	}

	#
	# requests for data other than index values and descriptions require additional processing
	#
	if ((($cacti_request == "query") || ($cacti_request == "get")) &&
		($data_request != "ups.device") &&
		($data_request != "ups.description")) {

		$ups_count = 0;

		foreach ($ups_array as $ups) {

			#
			# the Cacti data-query XML uses raw NUT commands, so just pass them through
			#
			fwrite($nut_session, "GET VAR " . $ups['index'] . " " . $data_request . "\n");

			$nut_response = fgets($nut_session, 1024);

			#
			# use regex to locate the requested data
			#
			# exit if no match found
			#
			# note response is limited to numeric data
			#
			if (preg_match("/^VAR \S+ \S+ \"(\d*\.?\d*)\"$/", $nut_response, $scratch) == 0) {

				echo ("FATAL: Unable to locate the requested data in NUT output\n");
				return;
			}

			#
			# matches were found so use them for the requested data
			#
			else {
				$ups_array[$ups_count]['data'] = trim($scratch[1]);
			}

			#
			# incrmenet the ups counter
			#
			$ups_count++;
		}
	}

	#
	# done collecting data so close the connection
	#
	fwrite($nut_session, "LOGOUT\n");
	fclose($nut_session);

	#
	# generate output
	#
	$ups_count = 0;

	foreach ($ups_array as $ups) {

		#
		# return output data according to $cacti_request
		#
		switch ($cacti_request) {

			#
			# if they want an index, dump a list of numbers
			#
			case "index":

				echo ($ups['index'] . "\n");
				break;

			#
			# if they want query data, give them the type they want
			#
			case "query":

				switch ($data_request) {

					case "ups.device":

						echo ($ups['index'] . ":" . $ups['index'] . "\n");
						break;

					case "ups.description":

						echo ($ups['index'] . ":" . $ups['desc'] . "\n");
						break;

					default:

						echo ($ups['index'] . ":" . $ups['data'] . "\n");
						break;
				}

				break;

			#
			# if they want explicit data, give them the type they want
			#
			case "get":

				switch ($data_request) {

					case "ups.device":

						echo ($ups['index'] . "\n");
						break;

					case "ups.description":

						echo ($ups['desc'] . "\n");
						break;

					default:

						if (isset($GLOBALS['called_by_script_server']) == TRUE) {

							return($ups['data']);
						}

						else {
							echo ($ups['data'] . "\n");
						}

						break;
				}

				break;
		}

		$ups_count++;
	}
}

#
# display the syntax
#
function ss_nut_ups_status_syntax() {

	echo ("syntax: ss_nut_ups_status.php <hostname>:[<nut_port>]:[<nut_timeout>] \ \n" .
	"      (index|query <fieldname>|get <fieldname> <ups_device>)\n");
}

?>
